#include "QuaternionDisplay.h"

#ifdef __APPLE__
#include <OpenGL/gl.h>
#else
#include <GL/gl.h>
#endif

#include <math.h>

#include "DisplayObjects.h"
#include "Matrix.h"
#include "Quaternion.h"
#include "Vector.h"

#define QUATERNION_CIRCLE_PRECISION 64
#define QUATERNION_ARROW_PRECISION 16

void drawQuaternion(Quaternion quaternion) {
	Vector front, up, right;
	Matrix matrix;
	int segmentIndex;
	float angle, currentAngle, nextAngle = 0.0f, currentAngleWidth, nextAngleWidth;
	Vector arrowAngleVector, arrowNormalVector;
	float magnitude;
	
	magnitude = sqrt((quaternion.x * quaternion.x) +
	                 (quaternion.y * quaternion.y) +
	                 (quaternion.z * quaternion.z) +
	                 (quaternion.w * quaternion.w));
	
	glPushMatrix();
	glScalef(magnitude, magnitude, magnitude);
	
	Quaternion_toAxisAngle(quaternion, &front, &angle);
	Vector_normalize(&front);
	
	if (isnan(Vector_magnitudeSquared(front))) {
		glEnable(GL_BLEND);
		glDisable(GL_DEPTH_TEST);
		glEnable(GL_NORMALIZE);
		
		drawSingleColoredSphere(0.0f, 1.0f, 0.0f, 0.25f);
		glScalef(0.1f, 0.1f, 0.1f);
		drawSingleColoredSphere(1.0f, 0.0f, 0.0f, 0.25f);
		
		glDisable(GL_NORMALIZE);
		glDisable(GL_BLEND);
		glEnable(GL_DEPTH_TEST);
		glPopMatrix();
		return;
	}
	
	up = Vector_withValues(0.0f, 1.0f, 0.0f);
	if (fabs(Vector_dot(up, front)) > 0.9999) {
		right = Vector_withValues(1.0f, 0.0f, 0.0f);
		up = Vector_cross(front, right);
		Vector_normalize(&up);
		right = Vector_cross(up, front);
	} else {
		right = Vector_cross(up, front);
		Vector_normalize(&right);
		up = Vector_cross(front, right);
	}
	matrix = Matrix_fromDirectionVectors(right, up, front);
	
	glMultMatrixf(matrix.m);
	
	glBegin(GL_QUADS);
	glColor3f(1.0f, 0.0f, 0.0f);
	glNormal3f(0.0f, 0.0f, -1.0f);
	glVertex3f(0.025f, -0.025f, -1.0f);
	glVertex3f(-0.025f, -0.025f, -1.0f);
	glVertex3f(-0.025f, 0.025f, -1.0f);
	glVertex3f(0.025f, 0.025f, -1.0f);
	
	glNormal3f(0.0f, 0.0f, 1.0f);
	glVertex3f(-0.025f, -0.025f, 0.55f);
	glVertex3f(0.025f, -0.025f, 0.55f);
	glVertex3f(0.025f, 0.025f, 0.55f);
	glVertex3f(-0.025f, 0.025f, 0.55f);
	
	glNormal3f(-1.0f, 0.0f, 0.0f);
	glVertex3f(-0.025f, -0.025f, -1.0f);
	glVertex3f(-0.025f, -0.025f, 0.55f);
	glVertex3f(-0.025f, 0.025f, 0.55f);
	glVertex3f(-0.025f, 0.025f, -1.0f);
	
	glNormal3f(0.0f, 1.0f, 0.0f);
	glVertex3f(-0.025f, 0.025f, -1.0f);
	glVertex3f(-0.025f, 0.025f, 0.55f);
	glVertex3f(0.025f, 0.025f, 0.55f);
	glVertex3f(0.025f, 0.025f, -1.0f);
	
	glNormal3f(1.0f, 0.0f, 0.0f);
	glVertex3f(0.025f, 0.025f, -1.0f);
	glVertex3f(0.025f, 0.025f, 0.55f);
	glVertex3f(0.025f, -0.025f, 0.55f);
	glVertex3f(0.025f, -0.025f, -1.0f);
	
	glNormal3f(0.0f, -1.0f, 0.0f);
	glVertex3f(0.025f, -0.025f, -1.0f);
	glVertex3f(0.025f, -0.025f, 0.55f);
	glVertex3f(-0.025f, -0.025f, 0.55f);
	glVertex3f(-0.025f, -0.025f, -1.0f);
	glEnd();
	
	arrowAngleVector.x = 0.0f;
	arrowAngleVector.y = 0.2f;
	arrowAngleVector.z = 0.4f;
	Vector_normalize(&arrowAngleVector);
	arrowNormalVector = Vector_cross(arrowAngleVector, Vector_withValues(1.0f, 0.0f, 0.0f));
	
	glBegin(GL_TRIANGLES);
	for (segmentIndex = 0; segmentIndex < QUATERNION_CIRCLE_PRECISION; segmentIndex++) {
		glNormal3f(arrowNormalVector.y * sin((M_PI * 2 * (segmentIndex + 1)) / (QUATERNION_CIRCLE_PRECISION - 1)), arrowNormalVector.y * cos((M_PI * 2 * (segmentIndex + 1)) / (QUATERNION_CIRCLE_PRECISION - 1)), arrowNormalVector.z);
		glVertex3f(sin((M_PI * 2 * (segmentIndex + 1)) / (QUATERNION_CIRCLE_PRECISION - 1)) * 0.2f, cos((M_PI * 2 * (segmentIndex + 1)) / (QUATERNION_CIRCLE_PRECISION - 1)) * 0.2f, 0.6f);
		glNormal3f(arrowNormalVector.y * sin((M_PI * 2 * segmentIndex) / (QUATERNION_CIRCLE_PRECISION - 1)), arrowNormalVector.y * cos((M_PI * 2 * segmentIndex) / (QUATERNION_CIRCLE_PRECISION - 1)), arrowNormalVector.z);
		glVertex3f(sin((M_PI * 2 * segmentIndex) / (QUATERNION_CIRCLE_PRECISION - 1)) * 0.2f, cos((M_PI * 2 * segmentIndex) / (QUATERNION_CIRCLE_PRECISION - 1)) * 0.2f, 0.6f);
		glNormal3f(arrowNormalVector.y * sin((M_PI * 2 * (segmentIndex + 0.5f)) / (QUATERNION_CIRCLE_PRECISION - 1)), arrowNormalVector.y * cos((M_PI * 2 * (segmentIndex + 0.5f)) / (QUATERNION_CIRCLE_PRECISION - 1)), arrowNormalVector.z);
		glVertex3f(0.0f, 0.0f, 1.0f);
		
		glNormal3f(0.0f, 0.0f, -1.0f);
		glVertex3f(sin((M_PI * 2 * segmentIndex) / (QUATERNION_CIRCLE_PRECISION - 1)) * 0.2f, cos((M_PI * 2 * segmentIndex) / (QUATERNION_CIRCLE_PRECISION - 1)) * 0.2f, 0.6f);
		glVertex3f(sin((M_PI * 2 * (segmentIndex + 1)) / (QUATERNION_CIRCLE_PRECISION - 1)) * 0.2f, cos((M_PI * 2 * (segmentIndex + 1)) / (QUATERNION_CIRCLE_PRECISION - 1)) * 0.2f, 0.6f);
		glVertex3f(0.0f, 0.0f, 0.6f);
	}
	glEnd();
	
	glBegin(GL_QUADS);
	glColor3f(0.0f, 1.0f, 0.0f);
	for (segmentIndex = 0; segmentIndex < QUATERNION_CIRCLE_PRECISION; segmentIndex++) {
		glNormal3f(sin((M_PI * 2 * segmentIndex) / (QUATERNION_CIRCLE_PRECISION - 1)), cos((M_PI * 2 * segmentIndex) / (QUATERNION_CIRCLE_PRECISION - 1)), 0.0f);
		glVertex3f(sin((M_PI * 2 * segmentIndex) / (QUATERNION_CIRCLE_PRECISION - 1)), cos((M_PI * 2 * segmentIndex) / (QUATERNION_CIRCLE_PRECISION - 1)), -0.1f);
		glVertex3f(sin((M_PI * 2 * segmentIndex) / (QUATERNION_CIRCLE_PRECISION - 1)), cos((M_PI * 2 * segmentIndex) / (QUATERNION_CIRCLE_PRECISION - 1)), 0.1f);
		glNormal3f(sin((M_PI * 2 * (segmentIndex + 1)) / (QUATERNION_CIRCLE_PRECISION - 1)), cos((M_PI * 2 * (segmentIndex + 1)) / (QUATERNION_CIRCLE_PRECISION - 1)), 0.0f);
		glVertex3f(sin((M_PI * 2 * (segmentIndex + 1)) / (QUATERNION_CIRCLE_PRECISION - 1)), cos((M_PI * 2 * (segmentIndex + 1)) / (QUATERNION_CIRCLE_PRECISION - 1)), 0.1f);
		glVertex3f(sin((M_PI * 2 * (segmentIndex + 1)) / (QUATERNION_CIRCLE_PRECISION - 1)), cos((M_PI * 2 * (segmentIndex + 1)) / (QUATERNION_CIRCLE_PRECISION - 1)), -0.1f);
	}
	glEnd();
	
	glBegin(GL_QUADS);
	glColor3f(0.0f, 0.0f, 1.0f);
	for (segmentIndex = 0; segmentIndex < (int) ((QUATERNION_CIRCLE_PRECISION * angle) / (M_PI * 2)); segmentIndex++) {
		currentAngle = (M_PI * 2 * segmentIndex) / QUATERNION_CIRCLE_PRECISION;
		nextAngle = (M_PI * 2 * (segmentIndex + 1)) / QUATERNION_CIRCLE_PRECISION;
		currentAngleWidth = 1.0f - currentAngle / angle;
		nextAngleWidth = 1.0f - nextAngle / angle;
		
		glNormal3f(sin(currentAngle), cos(currentAngle), 0.0f);
		glVertex3f(sin(currentAngle) * 0.99f, cos(currentAngle) * 0.99f, 0.02f + 0.08f * currentAngleWidth);
		glVertex3f(sin(currentAngle) * 0.99f, cos(currentAngle) * 0.99f, -0.02f + -0.08f * currentAngleWidth);
		glNormal3f(sin(nextAngle), cos(nextAngle), 0.0f);
		glVertex3f(sin(nextAngle) * 0.99f, cos(nextAngle) * 0.99f, -0.02f + -0.08f * nextAngleWidth);
		glVertex3f(sin(nextAngle) * 0.99f, cos(nextAngle) * 0.99f, 0.02f + 0.08f * nextAngleWidth);
		glNormal3f(sin(currentAngle), cos(currentAngle), 0.0f);
		glVertex3f(sin(currentAngle) * 1.01f, cos(currentAngle) * 1.01f, 0.02f + 0.08f * currentAngleWidth);
		glVertex3f(sin(currentAngle) * 1.01f, cos(currentAngle) * 1.01f, -0.02f + -0.08f * currentAngleWidth);
		glNormal3f(sin(nextAngle), cos(nextAngle), 0.0f);
		glVertex3f(sin(nextAngle) * 1.01f, cos(nextAngle) * 1.01f, -0.02f + -0.08f * nextAngleWidth);
		glVertex3f(sin(nextAngle) * 1.01f, cos(nextAngle) * 1.01f, 0.02f + 0.08f * nextAngleWidth);
	}
	
	currentAngle = nextAngle;
	currentAngleWidth = 1.0f - currentAngle / angle;
	
	glNormal3f(sin(currentAngle), cos(currentAngle), 0.0f);
	glVertex3f(sin(currentAngle) * 0.99f, cos(currentAngle) * 0.99f, 0.02f + 0.08f * currentAngleWidth);
	glVertex3f(sin(currentAngle) * 0.99f, cos(currentAngle) * 0.99f, -0.02f + -0.08f * currentAngleWidth);
	glNormal3f(sin(angle), cos(angle), 0.0f);
	glVertex3f(sin(angle) * 0.99f, cos(angle) * 0.99f, -0.02f);
	glVertex3f(sin(angle) * 0.99f, cos(angle) * 0.99f, 0.02f);
	glNormal3f(sin(currentAngle), cos(currentAngle), 0.0f);
	glVertex3f(sin(currentAngle) * 1.01f, cos(currentAngle) * 1.01f, 0.02f + 0.08f * currentAngleWidth);
	glVertex3f(sin(currentAngle) * 1.01f, cos(currentAngle) * 1.01f, -0.02f + -0.08f * currentAngleWidth);
	glNormal3f(sin(angle), cos(angle), 0.0f);
	glVertex3f(sin(angle) * 1.01f, cos(angle) * 1.01f, -0.02f);
	glVertex3f(sin(angle) * 1.01f, cos(angle) * 1.01f, 0.02f);
	glEnd();
	
	glPopMatrix();
}
